package com.ejie.ab04b.exception;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

/**
 * Excepcion para el servicio web de interoperabilidad con la policia, que
 * obtiene los datos de identidad de personas.
 * 
 * @author GFI-NORTE
 * 
 */
public class DatosIdentidadException extends RuntimeException implements Serializable {

	private static final long serialVersionUID = 1L;

	private final String codEstadoError;
	private final String codLiteralError;

	/**
	 * Instantiates a new datos identidad exception.
	 * 
	 *  throwable            Throwable
	 *
	 * @param throwable the throwable
	 */
	public DatosIdentidadException(Throwable throwable) {
		this(DatosIdentidadException.Error.SERV_NO_DISP.getCodError(), throwable);
	}

	/**
	 * Instantiates a new datos identidad exception.
	 * 
	 *  codEstadoError            String
	 *
	 * @param codEstadoError the cod estado error
	 */
	public DatosIdentidadException(String codEstadoError) {
		this(codEstadoError, null);
	}

	/**
	 * Instantiates a new datos identidad exception.
	 * 
	 *  codEstadoError            String
	 *  throwable            Excepcion ocurrida
	 *
	 * @param codEstadoError the cod estado error
	 * @param throwable the throwable
	 */
	public DatosIdentidadException(String codEstadoError, Throwable throwable) {
		super("DatosIdentidadException", throwable);
		this.codEstadoError = codEstadoError;
		this.codLiteralError = Error.getByCod(codEstadoError).getCodLiteralError();

	}

	/**
	 * Gets the cod estado error.
	 * 
	 *  String
	 *
	 * @return the cod estado error
	 */
	public String getCodEstadoError() {
		return this.codEstadoError;
	}

	/**
	 * Gets the cod literal error.
	 * 
	 *  String
	 *
	 * @return the cod literal error
	 */
	public String getCodLiteralError() {
		return this.codLiteralError;
	}

	/**
	 * Errores posibles en el servicio de datos de identidad de la policia
	 * 
	 * @author GFI-NORTE
	 */
	private static enum Error {
		VALIDACIONES("0000", "error.datosidentidadWS.validaciones"), //
		DATOS_INCORRECTOS("0226", "error.datosidentidadWS.datosincorrectos"), //
		NUMDOC_ERRONEO("0231", "error.datosidentidadWS.numdocumentoerroneo"), //
		MULT_COINCID("0232", "error.datosidentidadWS.multiplescoincidencias"), //
		DOC_NO_ENCONTRADO("68", "error.datosidentidadWS.documentonoencontrado"), //
		SIN_COINCIDENCIAS("0233", "error.datosidentidadWS.sincoincidencias"), //
		SERV_NO_DISP("0901", "error.datosidentidadWS.servicionodisponible"), //
		GENERICO("", "error.datosidentidadWS.generico") //
		;

		private final String codError;
		private final String codliteralError;

		private final static Map<String, DatosIdentidadException.Error> MAPERROR = new HashMap<String, DatosIdentidadException.Error>();
		static {
			for (DatosIdentidadException.Error item : DatosIdentidadException.Error.values()) {
				DatosIdentidadException.Error.MAPERROR.put(item.getCodError(), item);
			}
		}

		/**
		 * Instantiates a new error.
		 * 
		 *  codError            String
		 *  literalError            String
		 *
		 * @param codError the cod error
		 * @param literalError the literal error
		 */
		private Error(String codError, String literalError) {
			this.codError = codError;
			this.codliteralError = literalError;
		}

		/**
		 * Gets the by cod.
		 * 
		 *  codError            String
		 *  Error
		 *
		 * @param codError the cod error
		 * @return the by cod
		 */
		public static Error getByCod(String codError) {
			Error error = DatosIdentidadException.Error.MAPERROR.get(codError);
			return error != null ? error : DatosIdentidadException.Error.GENERICO;
		}

		/**
		 * Gets the cod error.
		 * 
		 *  String
		 *
		 * @return the cod error
		 */
		public String getCodError() {
			return this.codError;
		}

		/**
		 * Gets the cod literal error.
		 * 
		 *  String
		 *
		 * @return the cod literal error
		 */
		public String getCodLiteralError() {
			return this.codliteralError;
		}

	}
}
